unit FTPSession;
{
    UNIT FTPSession;
    Version number 1.02(beta)

This unit contains the FTP session manager TFTPSession class.
All methods are described in the interface part.

Version history:
    11th of July   : first release
    6th of August  : added GetTransferInfo() method as in TDataConnection.
    11th of August : corrected error handling.
    16th of August : added TerminateTransfer() method.

Created by Pter Karsai, 10-11th of July, 6th,11th,16th of August '99.

}
interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Forms, ExtCtrls,
     ControlConnection, DataConnection, Winsock, WSocket;

{ Session manager error codes }
const smNO_ERROR      = 0;
      smSOCKET_ERROR  = 1;
      smDISK_IO_ERROR = 2;
      smTIMEOUT_ERROR = 3;
      smNOT_CONNECTED = 4;

      smLOGIN_FAILED  = 5;
      smCOMMAND_ERROR = 6;
      smRECV_FAILED   = 7;
      smSEND_FAILED   = 8;
      smABORTED       = 9;
      smTERMINATE     = 10;

{ Use these constants as smSocketErrorDesc[smX*] }
const smErrorDesc: array[0..10] of string =
      ('OK.', 'Socket error: remote host unreachable, maybe network is down.',
       'Disk I/O error.', 'Timeout.', 'Not connected.', 'Login FAILED',
       'Command error.', 'Receiving file or directory list failed.',
       'Sending file failed.', 'Transfer ABORted by user.',
       'Termination by user.');

type TOutDial = procedure of object;  { type TOutDial simply call-type for
                                        external parameter-given procedures }

type EFCannotGetLocalIP = class(Exception);
{ thrown by Create() constructor if resolving local IP failed. }

type TFTPSession = class(TObject)

private
{ timeout values }
   ConnectTimeOut    : word;
   WelcomeMsgTimeOut : word;
   ReplyTimeOut      : word;

{ connection layers }
   ControlConnection : TControlConnection; { FTP control connection manager }
   DataConnection    : TDataConnection;    { FTP data connection manager layer }

{ miscellaneous data }
   Owner             : TComponent;  { owner of connection manager layers }
   LastErrorText     : string;      { last error text for GetLastErrorText() }
   LastErrorType     : byte;        { last error type for GetLastErrorType() }
   LastCommand       : string;      { last command sent }
   ServerReply       : TStringList; { server's reply - reference only. }
   LocalIP           : string;      { local IP address in '127,0,0,1' format }
   LocalIPDotted     : string;      { local IP address in dotted format }
   LocalName         : string;      { local host's name }
   DoAbort           : boolean;     { if TRUE, abort operation in progress }
   Terminating       : boolean;     { if TRUE, terminating is in progress }
   InSequenceS       : boolean;     { if TRUE, operation is in progress }
{ ----------------------------------------------------------------------------}
{ private methods }
{ ----------------------------------------------------------------------------}
   procedure CallExternal(ExtProc: TOutDial; ErrorType: byte; Msg: string);
{ Function: CallExternal call predefined external event-procedures if it's
  possible (if value assigned). The 'Msg' parameter will be saved to
  LastErrorText and it'll be accessible via GetLastErrorText(). Parameter
  'ErrorType' will be saved to LastErrorType and will accessible via
  GetLastErrorType(), too. }

   procedure SEHSendCommand(Command2Send: string);
{ Session's Error Handled SendCommand() }

   procedure SEHGetReply(TimeOut: word);
{ Session's Error Handled GetServerReply() }

   procedure UnexpectedReply;
{ Read unexpected reply and call external procedure }

   procedure SendTYPEPORT(TypeC: char; PortNo: word);
{ General TYPE-PORT sequence (made it to method 'cause it used two times and
  the TYPE&PORT handler code is about 30 lines) }

   function Receive(FileName, RFileName: string; AppendTo: boolean;
                    TypeC: char): byte;
{ General receiver sequence for LIST and RETR. }

public
{ ----------------------------------------------------------------------------}
{ external 'events' }
{ ----------------------------------------------------------------------------}
   OnCConConnected   : TOutDial;  { called when control connection built }
   OnCConCommand     : TOutDial;  { called when a control command sent }
   OnServerReply     : TOutDial;  { called when a server reply arrive }
   OnCConClosed      : TOutDial;  { called when control connection closed }
   OnCConError       : TOutDial;  { called when an error occur in control con. }
   OnDConConnected   : TOutDial;  { called when data connection built }
   OnDConClosed      : TOutDial;  { called when data connection closed }
   OnDConError       : TOutDial;  { called when an error occur in data con.}

{ ----------------------------------------------------------------------------}
{ FTP session management }
{ ----------------------------------------------------------------------------}
   function InSequence: boolean;
{ Function: Returns TRUE if operation is in progress, otherwise FALSE. }

   function ConnectToServer(RemoteServer: string; RemotePort: word): byte;
{ Function: Connect to the remote server. Returning value defined as table
  of smX* constants.
  Possibly can call OnCConConnected, OnCConError. }

   function Login(LoginName, Password: string): byte;
{ Function: Execute the login sequence. Returning value is an smX* constant.
  Possibly can call OnServerReply, OnConClosed, OnCConError. }

   function GetDirList(DirFile: string): byte;
{ Function: Get the list of the current directory. Transfer type ASCII, data
  will saved to file DirFile. Returning value as smX* constants. }

   function DownloadFile(FileName, RFileName: string; TypeC: char;
                         Append2: boolean): byte;
{ Function: Download 'RFileName' to 'FileName'. TypeC is the transfer type
  ('A' for ASCII, 'I' for Image[Binary]). If Append2 is TRUE, incoming data will
  appended to existing file (if exists). Returning value as smX* constants. }

   function UploadFile(FileName, RFileName: string; TypeC: char;
                       StartOffset: longint): byte;
{ Function: Upload local file FileName to remote server (file RFileName). If
  StartOffset higher than zero, command APPE will used instead of STOR (see
  docs). Returning value as smX* constants. }

   function RestartAt(RestAt: longint): byte;
{ Function: Set restart marker to byte RestAt. Returning value as smX*. }

   function GetSystemType: string;
{ Function: Returns with reply of SYST command, without starting numerical code.
  If there was an error, string will be empty. }

   function GetCurrentPath: string;
{ Function: Return with the current path on the FTP site. String will empty if
  an error occur. PWD command. }

   function ChangeWorkDir(ChgWorkDir2: string): byte;
{ Function: Changes the working directory, returning value as smX* constants. }

   function Abort: byte;
{ Function: Sends an ABOR command. }

   procedure SendNOOP;
{ Function: Sends a NOOP command. }

   procedure Close;
{ Function: Close the current FTP session. If no data connection alive, send
  a QUIT command. If no reply in ReplyTimeOut secs, close control and data
  connection both. This method automatically called with TFTPSession.Free. }

   function GetServerReply: TStringList;
{ Function: Returns with the third-party reference of the server reply. This
  method is not same as TControlConnection's GetServerReply()! It just return
  with reference of TControlConnection's ServerReply. You shall use this method
  if an OnServerReply event occur. }

   function GetLastErrorText: string;
{ Function: Returns with the description of last error provided by underlaying
  manager layers. }

   function GetLastErrorType: byte;
{ Function: Returns with last error type defined in table of smX* constants. }

   function GetLastCommand: string;
{ Function: Returns with the last command sent. }

   procedure GetLocalHost(var LName, IP, IPD: string);
{ Function: On return parameter LName will contain the local host name,
  parameter IP will contain local IP address in 'xxx,xxx,xxx,xxx' format and
  parameter IPD will contain local IP address in dotted format as
  '127.0.0.1'. }

   procedure GetTransferInfo(var TotalTransferedP, TransferedBytesP: longint);
{ Function: same as in TDataConnection }

   procedure TerminateTransfer;
{ Function: close data transfer if it's in progress. }

{ ----------------------------------------------------------------------------}
{ constructor and destructor }
{ ----------------------------------------------------------------------------}
    constructor Create(PConnectTimeOut, PWelcomeMsgTimeOut,
                       PReplyTimeOut: word; AOwner: TComponent);
{ PConnectTimeOut parameter used to set a time-out check while waiting for a
  connection. PWelcomeMsgTimeOut used to set time-out for the first message
  after the control connection connected (welcoming message). PReplyTimeOut
  used to set time-out limit for reply receiving.
  See TControlConnection's GetServerReply() method.

  Throws EFCannotGetLocalIP if resolving local IP failed. }

    destructor Destroy; override;
    procedure Free;
end;


implementation
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ Private methods -------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

procedure TFTPSession.CallExternal(ExtProc: TOutDial; ErrorType: byte;
                                   Msg: string);
begin
     if not Assigned(ControlConnection) then
     begin
        LastErrorType:= smTERMINATE;
        Exit
     end;
{ save error definition }
     LastErrorText:= Msg; LastErrorType:= ErrorType;
     if ErrorType <> smNO_ERROR then InSequenceS:= false;
{ call external handler if assigned }
     if Assigned(ExtProc) then ExtProc;
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.SEHSendCommand(Command2Send: string);
begin
     if not Assigned(ControlConnection) then
     begin
        LastErrorType:= smTERMINATE;
        Exit
     end;

{ call external proc. }
     if Copy(Command2Send, 1, 4) = 'PASS' then
        LastCommand:= 'PASS (hidden)'  { hide password if the command is PASS }
     else
        LastCommand:= Command2Send;
     CallExternal(OnCConCommand, smNO_ERROR, '');
{ try to send command }
     try
        ControlConnection.SendCommand(Command2Send);
     except
        on E: ECNotConnected do
           CallExternal(OnCConError, smNOT_CONNECTED, E.Message);
     end;

end;

{------------------------------------------------------------------------------}

procedure TFTPSession.SEHGetReply(TimeOut: word);
begin
     if not Assigned(ControlConnection) then
     begin
        LastErrorType:= smTERMINATE;
        Exit
     end;
{ if we aren't connected, we don't have to do it... }
     if not Assigned(ControlConnection) then
     begin
           CallExternal(OnCConError, smNOT_CONNECTED,
                        'Control connection closed.');
           exit;
     end;
{ try to get reply }
     try
        ServerReply:= ControlConnection.GetServerReply(TimeOut);
     except
     { if control socket not connected... }
        on E: ECNotConnected do
           CallExternal(OnCConError, smNOT_CONNECTED, E.Message);
     { if an socket exception occur... }
        on E: ECSocketException do
           CallExternal(OnCConError, smSOCKET_ERROR, E.Message);
     { if no reply arrive in TimeOut seconds }
        on E: ECReplyTimedOut do
           CallExternal(OnCConError, smTIMEOUT_ERROR, E.Message);
     end;

     if LastErrorType = smNO_ERROR then
           CallExternal(OnServerReply, smNO_ERROR, '');
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.UnexpectedReply;
begin
     SEHGetReply(ReplyTimeOut);
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.SendTYPEPORT(TypeC: char; PortNo: word);
var orgSS : boolean; { original sequence state }
begin
     orgSS:= InSequenceS; InSequenceS:= true;
{ try send TYPE first }
     SEHSendCommand('TYPE ' + TypeC);

{ get reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

{ if there was no error, check what kinda reply I've got }
     if LastErrorType = smNO_ERROR then
     { if server replied with error code }
        if ControlConnection.GetServerReplyType <> srPOS_COMPLETION_REPLY then
        { call error event handler with last line of server reply }
           CallExternal(OnCConError, smCOMMAND_ERROR, ServerReply.Strings[
                        ServerReply.Count - 1]);

{ PORT command 'IP1,IP2,IP3,IP4,P1,P2' }
     if LastErrorType = smNO_ERROR then SEHSendCommand(Format('PORT %s,%d,%d',
        [LocalIP, Hi(PortNo), Lo(PortNo)]));

{ get reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

     if LastErrorType = smNO_ERROR then
     { if server replied with error code, inform user }
        if not ControlConnection.GetServerReplyType in [srPOS_COMPLETION_REPLY,
           srPOS_INTERMEDIATE_REPLY] then
        { call error event handler with last line of server reply }
           CallExternal(OnCConError, smCOMMAND_ERROR, ServerReply.Strings[
                        ServerReply.Count - 1]);

     InSequenceS:= orgSS;
end;

{------------------------------------------------------------------------------}

function TFTPSession.Receive(FileName, RFileName: string; AppendTo: boolean;
                             TypeC: char): byte;
var localSrvPort: word; { local listening socket's port }
    fSize       : longint; { local file size }
begin
{ always wait for the last command to be finished }
     InSequenceS:= true;

{ create data connection management class }
     DataConnection:= TDataConnection.Create(FileName, Owner);
     DataConnection.OnDataConClosed:= OnDConClosed;

{ prepare receiving, set: don't append to file if exists }
     try
        localSrvPort:= DataConnection.PrepareReceiving(AppendTo,
                                                       OnDConConnected, fSize);
     except
     { if an socket exception occur (e.g. couldn't allocate port }
        on E: ETSocketException do
           CallExternal(OnDConError, smSOCKET_ERROR, E.Message);
        on E: EInOutError do { - can't create or open file... }
           CallExternal(OnDConError, smDISK_IO_ERROR, E.Message);
     end;

{ if everything is alright, send TYPE & PORT command }
     SendTYPEPORT(TypeC, localSrvPort);

{ if resuming... }
     if AppendTo then RestartAt(fSize);
     InSequenceS:= true;

{ send LIST command }
     if LastErrorType = smNO_ERROR then
        if RFileName <> '' then
           SEHSendCommand('RETR ' + RFileName)
        else
           SEHSendCommand('LIST');

{ get reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

{ if there was no error, check what kinda reply we've got }
     if LastErrorType = smNO_ERROR then
     { if server replied with error code }
        if ControlConnection.GetServerReplyType <> srPOS_PRELIMINARY_REPLY then
        { call error event handler with last line of server reply }
           CallExternal(OnCConError, smRECV_FAILED, ServerReply.Strings[
                        ServerReply.Count - 1]);

{ receive data }
     if LastErrorType = smNO_ERROR then begin
     { we're waiting for a reply when receiving finished }
        ControlConnection.SetExpectReply(true);
     { try to receive }
        try
           DataConnection.ReceiveFromServer(ConnectTimeOut);
        except
           on E: ETSocketException do  { socket error }
              CallExternal(OnDConError, smSOCKET_ERROR, E.Message);
           on E: EInOutError do { can't write file }
              CallExternal(OnDConError, smDISK_IO_ERROR, E.Message);
           on E: ETConnectingTimedOut do { server don't request connection }
              CallExternal(OnDConError, smTIMEOUT_ERROR, E.Message);
        end;
     end;

{ get reply - it should be an Positive Completion Reply! }
     if (LastErrorType = smNO_ERROR) then SEHGetReply(ReplyTimeOut);
  { if 426 Abort received... }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
         srNEG_TRANSIENT_COMP_RPY) and (ServerReply.Count = 1) and DoAbort then
     begin
         ControlConnection.SetExpectReply(true);
         SEHGetReply(ReplyTimeOut);
     end;

  { if 226 Abort successful received... }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY) and not DoAbort
     then
        Result:= smNO_ERROR
     else
        if DoAbort then
           Result:= smABORTED
        else
           Result:= smRECV_FAILED;

{ if we aborted the operation, don't let it to abort next... }
     DoAbort:= false; Terminating:= false;
     if Terminating then result:= smTERMINATE;
{ free DataConnection }
     if Assigned(DataConnection) then DataConnection.Free;
        DataConnection:= nil;

     InSequenceS:= false;
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ FTP session management methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
function TFTPSession.InSequence: boolean;
begin
    Result:= InSequenceS;  { return with internal sequence state }
end;

{------------------------------------------------------------------------------}

function TFTPSession.ConnectToServer(RemoteServer: string;
                                     RemotePort: word): byte;
begin
{ instanting TControlConnection class }
     ControlConnection:= TControlConnection.Create(RemoteServer,
                                                   IntToStr(RemotePort), Owner);

{ bind TControlConnection's 'external-procedure' event handlers }
     ControlConnection.OnUnexpectedReply:= UnexpectedReply;
     ControlConnection.OnControlClosed  := OnCConClosed;

{ try to connect to the remote host }
     try
        ControlConnection.ConnectToServer(ConnectTimeOut);
     except
     { if an socket exception occur, e.g. 'Connection refused' }
        on E: ECSocketException do
           CallExternal(OnCConError, smSOCKET_ERROR, E.Message);
     { if connecting just got timed out }
        on E: ECConnectingTimedOut do
           CallExternal(OnCConError, smTIMEOUT_ERROR, E.Message);
     end;

{ if there was no error, call OnCConConnected }
     if LastErrorType = smNO_ERROR then
           CallExternal(OnCConConnected, smNO_ERROR, 'Connected to server.');

{ get server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(WelcomeMsgTimeOut);

{ set returning value }
     Result:= LastErrorType;
end;

{------------------------------------------------------------------------------}

function TFTPSession.Login(LoginName, Password: string): byte;
begin
     InSequenceS:= true;
{ try to log in - at first, send 'USER<space>username' command. }
     if LastErrorType = smNO_ERROR then SEHSendCommand('USER ' + LoginName);

{ get reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

{ if there was no error, check what kinda reply I've got }
     if LastErrorType = smNO_ERROR then
     { if server replied with error code }
        if ControlConnection.GetServerReplyType in [srNEG_TRANSIENT_COMP_RPY,
           srNEG_PERMANENT_COMP_RPY] then
        { call error event handler with last line of server reply }
           CallExternal(OnCConError, smLOGIN_FAILED, ServerReply.Strings[
                        ServerReply.Count - 1]);

{ continue login sequence, with 'PASS<space>password' command. }
     if LastErrorType = smNO_ERROR then SEHSendCommand('PASS ' + Password);
{ get server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
{ if login completed, set result to smNO_ERROR, else set it to smLOGIN_FAILED }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY)
     then
        Result:= smNO_ERROR
     else
        Result:= smLOGIN_FAILED;

{ if transfer finished with terminate, free data connection }
     if Terminating then begin
        Result:= smTerminate;
        DataConnection.Free;
        DataConnection:= nil;
     end;

     InSequenceS:= false;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetDirList(DirFile: string): byte;
begin
{ receive directory list to DirFile in type ASCII (DirFile will overwritten) }
     Result:= Receive(DirFile, '', false, 'A');
end;

{------------------------------------------------------------------------------}

function TFTPSession.DownloadFile(FileName, RFileName: string; TypeC: char;
                                  Append2: boolean): byte;
begin
{ receive RFileName(remote) to FileName(local) }
     Result:= Receive(FileName, RFileName, Append2, TypeC);
end;

{------------------------------------------------------------------------------}

function TFTPSession.UploadFile(FileName, RFileName: string; TypeC: char;
                                StartOffset: longint): byte;
var localSrvPort: word; { local listening socket's port }
begin
     InSequenceS:= true;

{ create data connection management class }
     DataConnection:= TDataConnection.Create(FileName, Owner);
     DataConnection.OnDataConClosed:= OnDConClosed;
{ prepare receiving, set: don't append to file if exists }
     try
        localSrvPort:= DataConnection.PrepareSending(OnDConConnected);
     except
     { if an socket exception occur (e.g. couldn't allocate port }
        on E: ETSocketException do
           CallExternal(OnDConError, smSOCKET_ERROR, E.Message);
     end;

{ if everything is alright, send TYPE & PORT command }
     SendTYPEPORT(TypeC, localSrvPort);

{ send STOR or APPE command }
     if LastErrorType = smNO_ERROR then
        if StartOffset = 0 then
           SEHSendCommand('STOR ' + RFileName)
        else
           SEHSendCommand('APPE ' + RFileName);

{ get reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

{ if there was no error, check what kinda reply we've got }
     if LastErrorType = smNO_ERROR then
     { if server replied with error code }
        if ControlConnection.GetServerReplyType <> srPOS_PRELIMINARY_REPLY then
        { call error event handler with last line of server reply }
           CallExternal(OnCConError, smSEND_FAILED, ServerReply.Strings[
                        ServerReply.Count - 1]);

{ receive data }
     if LastErrorType = smNO_ERROR then begin
     { we're waiting for a reply when receiving finished }
        ControlConnection.SetExpectReply(true);
     { try to receive }
        try
           DataConnection.SendToServer(StartOffset, ConnectTimeOut);
        except
           on E: ETSocketException do  { socket error }
              CallExternal(OnDConError, smSOCKET_ERROR, E.Message);
           on E: EInOutError do { can't write file }
              CallExternal(OnDConError, smDISK_IO_ERROR, E.Message);
           on E: ETConnectingTimedOut do { server don't request connection }
              CallExternal(OnDConError, smTIMEOUT_ERROR, E.Message);
        end;
     end;

{ get reply - it should be an Positive Completion Reply! }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);

  { if 426 Abort received... }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
         srNEG_TRANSIENT_COMP_RPY) and (ServerReply.Count = 1) and DoAbort then
     begin
         ControlConnection.SetExpectReply(true);
         SEHGetReply(ReplyTimeOut);
     end;

  { if 226 Abort successful received... }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY) and not DoAbort
     then
        Result:= smNO_ERROR
     else
        if DoAbort then
           Result:= smABORTED
        else
           Result:= smRECV_FAILED;

{ if we aborted the operation, don't let it to abort next... }
     DoAbort:= false;
     if Terminating then Result:= smTerminate;
{ free DataConnection }
     if Assigned(DataConnection) then DataConnection.Free;
        DataConnection:= nil;

     InSequenceS:= false;
end;

{------------------------------------------------------------------------------}

function TFTPSession.RestartAt(RestAt: longint): byte;
begin
     InSequenceS:= true;

{ try to send REST command }
     SEHSendCommand(Format('REST %d', [RestAt]));
{ receive server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
{ look at it }
     if (LastErrorType = smNO_ERROR) and ((ControlConnection.GetServerReplyType=
        srPOS_COMPLETION_REPLY) or (ControlConnection.GetServerReplyType =
        srPOS_INTERMEDIATE_REPLY))
     then Result:= smNO_ERROR
     else Result:= smCOMMAND_ERROR;

     if Terminating then Result:= smTerminate;
     InSequenceS:= false;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetSystemType: string;
begin
     InSequenceS:= true;

{ send SYST command }
     SEHSendCommand('SYST');
{ receive server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
{ look at it }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY)
     then Result:= Copy(ServerReply.Strings[ServerReply.Count - 1], 4, 255)
     else Result:= '';

     InSequenceS:= false;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetCurrentPath: string;
var slRpy: string;  { last line of the server's reply }
begin
     InSequenceS:= true;

{ send PWD command }
     SEHSendCommand('PWD');
{ receive server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
{ look at it - format '200 "currentdir" is blahblah' }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY)
     then begin
          slRpy:= ServerReply.Strings[ServerReply.Count - 1];
          Result:= Copy(slRpy, 6, 255);
          SetLength(Result, Pos('"', Result) - 1);
     end
     else Result:= '';

     InSequenceS:= false;
end;
{------------------------------------------------------------------------------}

function TFTPSession.ChangeWorkDir(ChgWorkDir2: string): byte;
begin
     InSequenceS:= true;
{ send CWD command }
     SEHSendCommand('CWD ' + ChgWorkDir2);
{ get server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
{ if command successfully completed, set result to smNO_ERROR }
     if (LastErrorType = smNO_ERROR) and (ControlConnection.GetServerReplyType =
        srPOS_COMPLETION_REPLY)
     then
        Result:= smNO_ERROR
     else
        Result:= smCOMMAND_ERROR;

     if Terminating then Result:= smTerminate;
     InSequenceS:= false;
end;

{------------------------------------------------------------------------------}

function TFTPSession.Abort: byte;
begin
     InSequenceS:= true;
     if DataConnection.CurrentOperation = otSending then
        DataConnection.AbortOperation;
{ try to send ABOR command. }
     if LastErrorType = smNO_ERROR then SEHSendCommand('ABOR');
{ Receive() and Upload() will catch the server's reply and handle it }
{ if we done abort... or not, close data connection }
     DoAbort:= true;
     if Terminating then Result:= smTerminate;
     InSequenceS:= false;
 end;

{------------------------------------------------------------------------------}

procedure TFTPSession.SendNOOP;
var orgSS : boolean; { original sequence state }
begin
     orgSS:= InSequenceS; InSequenceS:= true;
{ try to send ABOR command - now without IP and Synch signals. }
     SEHSendCommand('NOOP');
{ get server's reply }
     if LastErrorType = smNO_ERROR then SEHGetReply(ReplyTimeOut);
     InSequenceS:= orgSS;
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.Close;
begin
     Terminating:= true;
{ both controlconnection and dataconnection have a shutdown() method in their
  Free() method, so don't care about losing system resources. }
     if Assigned(ControlConnection) then ControlConnection.Free;
        ControlConnection:= nil;
     if Assigned(DataConnection) then
     begin
        DataConnection.AbortOperation;
        DataConnection.Free;
     end;
        DataConnection:= nil;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetServerReply: TStringList;
begin
    if Assigned(ControlConnection) then Result:= ServerReply;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetLastErrorText: string;
begin
    Result:= LastErrorText;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetLastErrorType: byte;
begin
     Result:= LastErrorType;
end;

{------------------------------------------------------------------------------}

function TFTPSession.GetLastCommand: string;
begin
     Result:= LastCommand;
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.GetLocalHost(var LName, IP, IPD: string);
begin
     LName:= LocalName; IP:= LocalIP; IPD:= LocalIPDotted;
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.GetTransferInfo(var TotalTransferedP,
                                      TransferedBytesP: longint);
begin
     if Assigned(DataConnection) then
        DataConnection.GetTransferInfo(TotalTransferedP, TransferedBytesP);
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.TerminateTransfer;
begin
{ if no data connection alive, just simply exit }
    if not Assigned(DataConnection) then exit;
{ abort data operations }
    Terminating:= true;
    DataConnection.AbortOperation;
    { Receive() and UploadFile() will shut down DataConnection }
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ constructor/destructor methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
constructor TFTPSession.Create(PConnectTimeOut, PWelcomeMsgTimeOut,
                               PReplyTimeOut: word; AOwner: TComponent);
var pointPos   : byte;     { position of '.' sign }
begin
     inherited Create;
{ get local host name}
     try
        LocalName:= LocalHostName;
     except
        on ESocketException do
           raise EFCannotGetLocalIP.Create('Resolving of local IP failed.');
     end;

     LocalIP:= LocalIPList[0]; { first net interface's IP address }
     LocalIPDotted:= LocalIPList[0]; { same }

{ change all '.' to ',' for PORT command }
     while Pos('.', LocalIP) > 0 do
     begin
        pointPos:= Pos('.', LocalIP);
        Delete(LocalIP, pointPos, 1);
        Insert(',', LocalIP, pointPos);
     end;

{ clear event handlers to avoid bad reference-calls }
     OnCConConnected  := nil;
     OnCConCommand    := nil;
     OnServerReply    := nil;
     OnCConClosed     := nil;
     OnCConError      := nil;
     OnDConConnected  := nil;
     OnDConClosed     := nil;
     OnDConError      := nil;

{ save parameters }
     ConnectTimeOut    := PConnectTimeOut;
     WelcomeMsgTimeOut := PWelcomeMsgTimeOut;
     ReplyTimeOut      := PReplyTimeOut;
     Owner             := AOwner;

{ set data to default state }
     LastCommand       := '';
     LastErrorText     := '';
     LastErrorType     := smNO_ERROR;

{ we aren't in sequence }
     InSequenceS       := false;
     Terminating       := false;
end;

{------------------------------------------------------------------------------}

destructor TFTPSession.Destroy;
begin
{ free underlaying layers }
     Close;
     inherited Destroy;
end;

{------------------------------------------------------------------------------}

procedure TFTPSession.Free;
begin
     if Self <> nil then Destroy;
end;

end.
